package ga.core.validation;

import ga.core.individual.IIndividual;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Validator implementation that uses a set of rules.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class RuleValidator<T extends IIndividual<T>> implements IValidator<T> {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(RuleValidator.class
      .getName());

  /**
   * These rules must match with {@link #mandatoryThreshold}.
   */
  private final List<IValidationRule<T>> mandatoryRules = new ArrayList<IValidationRule<T>>();

  /**
   * These rules must match with {@link #proposedThreshold}.
   */
  private final List<IValidationRule<T>> proposedRules = new ArrayList<IValidationRule<T>>();

  /**
   * These rules must match with {@link #perfectionThreshold}.
   */
  private final List<IValidationRule<T>> perfectionRules = new ArrayList<IValidationRule<T>>();

  private final float mandatoryThreshold = 1f;
  private final float proposedThreshold = 0.5f;
  private final float perfectionThreshold = 0.3f;

  /**
   * Adds a rule to the set of rules.
   * 
   * @param rule
   *          Rule to add.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void addRule(final IValidationRule<T> rule) {
    switch (rule.getType()) {
    case MANDATORY:
      mandatoryRules.add(rule);
      break;
    case PROPOSED:
      proposedRules.add(rule);
      break;
    case PERFECTION:
      perfectionRules.add(rule);
      break;
    }

    if (LOGGER.isLoggable(Level.FINE)) {
      LOGGER.fine("added rule " + rule.getType() + " "
          + rule.getClass().getName());
    }
  }

  @Override
  public boolean isValid(final T individual, final GAContext context) {
    for (final IValidationRule<T> rule : mandatoryRules) {
      if (rule.matchingDegree(individual, context) < mandatoryThreshold) {
        if (LOGGER.isLoggable(Level.FINE)) {
          LOGGER.fine("individual did not match rule " + rule);
        }
        return false;
      }
    }

    for (final IValidationRule<T> rule : proposedRules) {
      if (rule.matchingDegree(individual, context) < proposedThreshold) {
        if (LOGGER.isLoggable(Level.FINE)) {
          LOGGER.fine("individual did not match rule " + rule);
        }
        return false;
      }
    }

    for (final IValidationRule<T> rule : perfectionRules) {
      if (rule.matchingDegree(individual, context) < perfectionThreshold) {
        if (LOGGER.isLoggable(Level.FINE)) {
          LOGGER.fine("individual did not match rule " + rule);
        }
        return false;
      }
    }

    return true;
  }
}
